//*************************************************************************************************
//
//	Description:
//		road_2layer.fx - Painted road shader for The Race. Supports two layers, each with full d/s/n
//										 mapping and a thickness map to simulate the look of painted lines / signs
//										 on the underlying surface.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     05/18/2006  1.0						Created
//		TNettleship			06/26/2006	1.1						Added ambient occlusion term from vertex colour,
//																							added option for compressed normals. Changes to make
//																							it work in the engine.
//		TNettleship			10/03/2006	1.2						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/09/2006	1.3						Fixed the texture mapping channel assignment in the engine.
//		TNettleship			10/17/2006	1.4						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.5						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.6						Fixed a bug with directional lighting.
//		TMann						11/24/2006	1.9						Added _WIN_GL_ support
//		TMann						11/27/2006	1.10					Texture filtering for GL
//		TMann						12/11/2006	1.12					PS3 version
//		TNettleship			02/01/2007	1.13					Pixel shader optimisations
//		BIrvine					02/13/2007	1.14					Temporary shadow implementation
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship			07/24/2007	2.02					Added anisotropic filtering to diffuse maps.
//		TNettleship			10/23/2007	2.03					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

#endif

//
// Textures
//

//
// Layer 1 - base tarmac + cracks etc.
//

#ifdef _3DSMAX_
texture layer1DiffuseTexture : DiffuseMap			// Diffuse colour in RGB
#else
texture layer1DiffuseTexture : TEXTURE				// Diffuse colour in RGB
#endif
<
	string UIName = "Layer1 Diffuse";
	bool appEdit = true;
>;

texture layer1SpecularTexture : TEXTURE				// Specular colour in RGB, shininess in alpha
<
	string UIName = "Layer1 Specular";
	bool appEdit = true;
>;

texture layer1NormalTexture : TEXTURE					// Normal map
<
	string UIName = "Layer1 Normal";
	bool appEdit = true;
>;

float layer1MinSpecPower
<
	string UIName = "Min Spec Power L1";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float layer1MaxSpecPower
<
	string UIName = "Max Spec Power L1";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float layer1GlobalSpecularFactor
<
	string UIName = "Specular Factor L1";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

//
// Layer 2 - lines + regular wear / dirty map
//

SPECIALISATION_PARAM( useLayer2, "Use Layer 2?", "USE_LAYER_2" )	// TRUE if layer 2 should be used
DECLARE_DEPENDENT_VERTEX_STREAM( texCoord1Dependency, texCoord1, TEXCOORD2, useLayer2 )

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
DEPENDENT_TEXTURE_PARAM( layer2DiffuseTexture, "L2 Diff {UV2}", useLayer2 )
DEPENDENT_TEXTURE_PARAM( layer2SpecularTexture, "L2 Spec {UV2}", useLayer2 )
DEPENDENT_TEXTURE_PARAM( layer2ThicknessTexture, "L2 Thick {UV2}", useLayer2 )
#endif

SPECIALISATION_PARAM( useLayer2NormalMap, "Use Layer2 normal map?", "USE_LAYER_2_NORMAL" )	// TRUE if layer 2 should be used

#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_LAYER_2_NORMAL ) )
BOTH_DEPENDENT_TEXTURE_PARAM( layer2NormalTexture, "Layer2 Normals", useLayer2;useLayer2NormalMap )	// only if both layer 2 enabled and layer 2 normals enabled
#endif

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer2MinSpecPower, "Min Spec Power L2", useLayer2, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer2MaxSpecPower, "Max Spec Power L2", useLayer2, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( layer2GlobalSpecularFactor, "Specular Factor L2", useLayer2, 0.0f, 1.0f, 1.0f )
#endif


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D layer1DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer1DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < layer1DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D layer1SpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer1SpecularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer1SpecularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D layer1NormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer1NormalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer1NormalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
sampler2D layer2DiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2DiffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
>
= sampler_state
{
	Texture = < layer2DiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D layer2SpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2SpecularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer2SpecularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D layer2ThicknessMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2ThicknessTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer2ThicknessTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif		// #if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )


#if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_LAYER_2_NORMAL ) )
sampler2D layer2NormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="layer2NormalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < layer2NormalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif		// #if defined( _3DSMAX_ ) || ( defined( USE_LAYER_2 ) && defined( USE_LAYER_2_NORMAL ) )


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position		: POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour			: TEXCOORD1;												// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV channel 1 texture coord
	float2 texCoord1	: TEXCOORD2;												// UV channel 2 texture coord
#else
	float4 colour			: COLOR;														// Vertex colour
	float2 texCoord0	: TEXCOORD1;												// UV channel 2 texture coord - N.B. swapped as this makes for a more natural mapping for the artists.
#if defined( USE_LAYER_2 )
	float2 texCoord1	: TEXCOORD0;												// UV channel 1 texture coord
#endif
#endif
	float3 normal			: NORMAL;														// Object space normal
	float3 tangent		: TANGENT;													// Object space tangent
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
#endif
	float3 tangent		: TEXCOORD2;												// Tangent vector (world space)
	float3 normal			: TEXCOORD3;												// Normal vector (world space)
	float4 eye				: TEXCOORD4;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT RoadVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour		= _input.colour;
	_output.texCoord0 = _input.texCoord0;
	DEPENDENT_CODE_START( useLayer2 )
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		_output.texCoord1 = _input.texCoord1;
#endif
	DEPENDENT_CODE_END( useLayer2 )

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal   = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;

	// Calculate object-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD5;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float3 tangent		: TEXCOORD2;												// Tangent vector (world space)
	float3 normal			: TEXCOORD3;												// Normal vector (world space)
	float4 eye				: TEXCOORD4;												// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD5_centroid;						// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
#if defined( USE_LAYER_2 )
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
#endif
	float3 tangent		: TEXCOORD2_centroid;								// Tangent vector (world space)
	float3 normal			: TEXCOORD3_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD4_centroid;								// Eye vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT RoadFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 layer1DiffuseTexColour = tex2D( layer1DiffuseMap, _input.texCoord0 );
	float4 layer1SpecularTexColour = tex2D( layer1SpecularMap, _input.texCoord0 );
	float4 normalMapColour = tex2D( layer1NormalMap, _input.texCoord0 );

	// Create tangent space vectors
	float3 TSnormal = normalize( _input.normal );
	float3 tangent = normalize( _input.tangent );
	float3 binormal = cross( TSnormal, tangent );
	float3 eye = normalize( _input.eye.xyz );

	// Fetch and decode the layer 1 map normal
	float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

	// Perturb the tangent space normal by the normal map
	float3 layer1Normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
	layer1Normal = normalize( layer1Normal );

	// If normal map support is required for layer 2
	// Interpolate between the layer1 and layer2 values to feed into the lighting system
	float4 diffuseTexColour;
	float4 specularTexColour;
	float3 normal;
	float minSpecPowerValue;
	float maxSpecPowerValue;
	float globalSpecularFactorValue;

	DEPENDENT_CODE_START(useLayer2)
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		float4 layer2DiffuseTexColour = tex2D( layer2DiffuseMap, _input.texCoord1 );
		float4 layer2SpecularTexColour = tex2D( layer2SpecularMap, _input.texCoord1 );
		float4 layer2ThicknessTexColour = tex2D( layer2ThicknessMap, _input.texCoord1 );
		float3 layer2Normal;

		DEPENDENT_CODE_START(useLayer2NormalMap)
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2_NORMAL )
			// Fetch and decode the layer 2 map normal
			normalMapColour = tex2D( layer2NormalMap, _input.texCoord1 );
			normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

			// Perturb the tangent space normal by the normal map
			layer2Normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
			layer2Normal = normalize( layer2Normal );
#endif
		DEPENDENT_CODE_ELSE(useLayer2NormalMap)
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_2_NORMAL )
		// No normal map, so use interpolated normal and constant specular strength
			layer2Normal = TSnormal;
#endif
		DEPENDENT_CODE_END(useLayer2NormalMap)

		float blend = layer2DiffuseTexColour.a;
		float thickness = layer2ThicknessTexColour.g;

		diffuseTexColour = lerp( layer1DiffuseTexColour, layer2DiffuseTexColour, blend );
		specularTexColour = lerp( layer1SpecularTexColour, layer2SpecularTexColour, blend );
		normal = lerp( layer1Normal, layer2Normal, thickness );
		minSpecPowerValue = lerp( layer1MinSpecPower, layer2MinSpecPower, blend );
		maxSpecPowerValue = lerp( layer1MaxSpecPower, layer2MaxSpecPower, blend );
		globalSpecularFactorValue = lerp( layer1GlobalSpecularFactor, layer2GlobalSpecularFactor, blend );
#endif
	DEPENDENT_CODE_ELSE( useLayer2 )
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_2 )
		diffuseTexColour = layer1DiffuseTexColour;
		specularTexColour = layer1SpecularTexColour;
		normal = layer1Normal;
		minSpecPowerValue = layer1MinSpecPower;
		maxSpecPowerValue = layer1MaxSpecPower;
		globalSpecularFactorValue = layer1GlobalSpecularFactor;
#endif
	DEPENDENT_CODE_END( useLayer2 )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



//
// Low Detail Shaders
//


// Output structure
struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD2;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
#endif
};




VSOUTPUT_LD RoadLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord0 = _input.texCoord0;
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
	_output.texCoord1 = _input.texCoord1;
#endif
	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	float4	diffuseTexColour;
	diffuseTexColour = _input.colour;

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

PSOUTPUT RoadLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 layer1DiffuseTexColour = tex2D( layer1DiffuseMap, _input.texCoord0 );

	float4 diffuseTexColour;

	DEPENDENT_CODE_START(useLayer2)
#if defined( _3DSMAX_ ) || defined( USE_LAYER_2 )
		float4 layer2DiffuseTexColour = tex2D( layer2DiffuseMap, _input.texCoord1 );
		float4 layer2ThicknessTexColour = tex2D( layer2ThicknessMap, _input.texCoord1 );

		float blend = layer2DiffuseTexColour.a;
		float thickness = layer2ThicknessTexColour.g;

		diffuseTexColour = lerp( layer1DiffuseTexColour, layer2DiffuseTexColour, blend );
#endif
	DEPENDENT_CODE_ELSE(useLayer2)
#if defined( _3DSMAX_ ) || !defined( USE_LAYER_2 )
		diffuseTexColour = layer1DiffuseTexColour;
#endif
	DEPENDENT_CODE_END(useLayer2)

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Road
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Road";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Road_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx RoadVertexShader();
		PixelShader = compile sce_fp_rsx RoadFragmentShader();
#else		
		VertexShader = compile vs_3_0 RoadVertexShader();
		PixelShader = compile ps_3_0 RoadFragmentShader();
#endif		
	}
}

technique Road_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Road_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx RoadLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx RoadLowDetailFragmentShader();
#else		
		VertexShader = compile vs_3_0 RoadLowDetailVertexShader();
		PixelShader = compile ps_3_0 RoadLowDetailFragmentShader();
#endif		
	}
}
